/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

        Input, output (export) functions for keydbs

        $Id: pgpKeyIO.c,v 1.62.2.3 2001/05/09 20:42:24 hal Exp $
____________________________________________________________________________*/

#include "pgpConfig.h"

#if HAVE_UNISTD_H
#include <unistd.h>
#endif


#include "pgpContext.h"
#include "pgpKeyPriv.h"
#include "pgpDebug.h"
#include "pgpAnnotate.h"
#include "pgpEncodePriv.h"
#include "pgpEnv.h"
#include "pgpErrors.h"
#include "pgpFileNames.h"
#include "pgpFileRef.h"
#include "pgpFileSpec.h"
#include "pgpPipeline.h"
#include "pgpRandomX9_17.h"
#include "pgpRandomPoolPriv.h"
#include "pgpRndSeed.h"
#include "pgpSigSpec.h"
#include "pgpPubKey.h"
#include "pgpTimeDate.h"
#include "pgpTrustPriv.h"
#include "pgpOptionList.h"
#include "pgpUtilitiesPriv.h"
#include "pgpX509Priv.h"

#if PGP_MACINTOSH
#include "MacStrings.h"
#include "MacFiles.h"
#endif


#define elemsof(x) ((unsigned)(sizeof(x)/sizeof(*x)))



/* See if the newly opened keyring appears to needs sig checking */
	static PGPError
sIsSigCheckNeeded( PGPKeyDB *db, PGPBoolean *sigCheckNeeded )
{
	PGPBoolean			needSigCheck;
	PGPKeyDBObj *		key;
	PGPKeyDBObj *		child;

	*sigCheckNeeded = FALSE;
	needSigCheck = FALSE;

	/*
	 * We look for a key with a revocation signature which has not
	 * been tried, such that the key's revoke bit is not set.  This is
	 * how some old version of PGP leave revocations.  We always set the
	 * key's revocation bit in the file, so this will not happen with
	 * keyrings written by this library.
	 */
	for( key = db->firstKeyInDB; IsntNull(key); key = key->next )
	{
		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		for( child = key->down; IsntNull(child); child = child->next )
		{
			if( !pgpKeyDBObjIsReal( child ) )
				continue;
			if ( pgpObjectType( child ) == RINGTYPE_SIG  &&
				 pgpSigType( child ) == PGP_SIGTYPE_KEY_REVOKE &&
				 pgpSigTrust( child ) == PGP_SIGTRUST_UNTRIED )
			 {
				if ( pgpObjectType( key ) == RINGTYPE_KEY &&
					 !pgpKeyRevoked( key ) )
				{
					/* Here we have our case we are looking for */
					needSigCheck = TRUE;
					break;
				}
			}
		}
	}

	*sigCheckNeeded = needSigCheck;
	return kPGPError_NoErr;
}


	PGPError
pgpOpenKeyDBFile_internal( 
	PGPContextRef			cdkContext,
	PGPOpenKeyDBFileOptions	openFlags,
	PFLFileSpecRef			pubFileRef,
	PFLFileSpecRef			privFileRef,
	PGPKeyDBRef *			keyDBOut )
{
	PGPKeyDB	   *db = NULL;
	PGPBoolean		sigCheckNeeded;
	PGPError		err;
	PFLFileInfo		pubFileInfo;
	PGPTime			curTime, pubModTime;

	/*
	 * Check creation time of pub file so we can check for objects
	 * expired since then.
	 */
	err = PFLGetFileInfo( pubFileRef, &pubFileInfo );
	if( IsPGPError( err ) )
	{
		err = kPGPError_NoErr;
		pubModTime = 0UL;
	} else {
		pubModTime = PGPGetPGPTimeFromStdTime( pubFileInfo.modificationTime );
	}

	db = pgpCreateKeyDBFromKeyRings( cdkContext, pubFileRef, privFileRef,
									 openFlags, &err );
	if( IsPGPError( err ) )
		goto error;

	/*
	 * Some earlier versions of PGP don't cache revocation info.  We will
	 * check signatures if the keyring has unchecked revocation signatures
	 * where the key does not have the revoke flag cached.
	 */
	if( IsPGPError( err = sIsSigCheckNeeded( db, &sigCheckNeeded ) ) )
		goto error;
	if( sigCheckNeeded )
	{
		if ( IsPGPError( err = PGPCheckKeyRingSigs( pgpKeyDBPeekRootSet(db),
													NULL, FALSE, NULL, NULL )))
			goto error;
		if ( IsPGPError( err = PGPCalculateTrust( db->rootSet, NULL ) ) )
			goto error;
	} else {
		/* We will re-run trust propagation if anything has expired */
		curTime = PGPGetTime();
		if( pgpKeyDBHasExpiringObjects( db, pubModTime, curTime ) ) {
			if ( IsPGPError( err = PGPCalculateTrust( db->rootSet, NULL ) ) )
				goto error;
		}
	}

	err = kPGPError_NoErr;
	
error:
	if (db != NULL && IsPGPError( err ))
	{
		PGPFreeKeyDB(db);
		db = NULL;
	}
	*keyDBOut	= db;
	
	pgpAssertErrWithPtr( err, *keyDBOut );
	return err;
}


/*
 * Open the specified keyrings for user, return keyset for it.
 */
	PGPError
PGPOpenKeyDBFile( 
	PGPContextRef			cdkContext,
	PGPOpenKeyDBFileOptions	openFlags,
	PGPFileSpecRef			pubFileRefIn,
	PGPFileSpecRef			privFileRefIn,
	PGPKeyDBRef *			keyDBOut )
{
	PFLFileSpecRef	pubFileRef	= (PFLFileSpecRef)pubFileRefIn;
	PFLFileSpecRef	privFileRef	= (PFLFileSpecRef)privFileRefIn;
	PGPUInt32		kdbid;
	PGPUInt32		numKeys;
	PGPUInt32 *		keyArray;
	PGPSize			keyArraySize;
	PGPError		err;

	PGPValidatePtr( keyDBOut );
	*keyDBOut	= NULL;
	PGPValidateContext( cdkContext );
	PFLValidateFileSpec( pubFileRef );
	if( IsntNull( privFileRef ) )
		PFLValidateFileSpec( privFileRef );
	
	pgpEnterPGPErrorFunction();

	/* In production mode, if no server, don't use double data structs */
#if !PGP_FORCEBACKEND
	if( !pgpRPCEnabled() )
	{
		/* Open it locally */
		return pgpOpenKeyDBFile_internal( cdkContext, openFlags, pubFileRef,
										  privFileRef, keyDBOut );
	}
#endif

	err = pgpOpenKeyDBFile_back( cdkContext, openFlags, pubFileRef,
								 privFileRef, &kdbid, &numKeys, &keyArray,
								 &keyArraySize );
	if( IsPGPError( err ) )
		return err;

	err = pgpNewFrontEndKeyDB( cdkContext, kdbid, keyArray, numKeys, keyDBOut);
	if( IsntNull( keyArray) )
		PGPFreeData( keyArray );

	return err;
}



/*
 * Add keys to a keydb from a dynamically allocated binary key buffer.
 * Makes a copy of the binary key buffer data, so caller can dispose of
 * it after this call.
 */
	PGPError
pgpImportKeyBinary_internal (
	PGPContextRef	cdkContext,
	PGPByte		   *buffer,
	size_t			length,
	PGPKeyDBRef *	outRef
	)
{
	PGPKeyDBRef		kdb;
	PGPError		err	= kPGPError_NoErr;

	*outRef	= NULL;

	/* Create a file type KeyDB from the buffer */
	kdb = pgpCreateKeyDBFromMemory (cdkContext, buffer, length, &err );
	if ( IsNull( kdb ) )
	{
		pgpAssert( IsPGPError( err ) );
	}
	*outRef	= kdb;
	
	return err;
}


/* This is only called from the front end */
	PGPError
pgpImportKeyBinary (
	PGPContextRef	cdkContext,
	PGPByte		   *buffer,
	size_t			length,
	PGPKeyDBRef *	outRef
	)
{
	PGPUInt32		kdbid;
	PGPUInt32		numKeys;
	PGPUInt32 *		keyArray;
	PGPSize			keyArraySize;
	PGPError		err	= kPGPError_NoErr;

	err = pgpImportKeyBinary_back( cdkContext, buffer, length, &kdbid,
								   &numKeys, &keyArray, &keyArraySize );
	
	if( IsPGPError( err ) )
		return err;

	err = pgpNewFrontEndKeyDB( cdkContext, kdbid, keyArray, numKeys, outRef);
	if( IsntNull( keyArray) )
		PGPFreeData( keyArray );
	
	return err;
}


/* Import an x509 cert from the specified optionlist input */
	static PGPError
sImportX509Certificate( PGPContextRef context, PGPKeyDBRef *db,
	PGPInputFormat inputFormat, PGPOptionListRef optionList)
{
	PGPByte		   *bufPtr;
	PGPSize			bufLength;
	PGPByte		   *outBuf=NULL, *certSet=NULL, *crlSet=NULL;
	PGPSize			outBufLength, certSetLength, crlSetLength;
	PGPBoolean		mustFreeBuf = FALSE;
	PGPKeyDB	   *db2 = NULL;
	char		   *passphrase;
	PGPSize			passphraseLength = 0;
	PGPKeyDBRef		refdb;
	PGPError		err = kPGPError_NoErr;
	
	pgpAssert(IsntNull( db ) );
	*db = NULL;

	err = pgpSetupInputToBuffer( context, optionList, &bufPtr, &bufLength,
								 &mustFreeBuf );
	if( IsPGPError( err ) )
		goto error;

	/* Refdb helps on input */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_KeyDBRef, FALSE,
						 "%p", &refdb ) ) )
		goto error;

	if( inputFormat == kPGPInputFormat_PKCS12 ||
		inputFormat == kPGPInputFormat_PrivateKeyInfo )
	{
		/* Input a private X.509 key */
		PGPKeyDBObj *key = NULL;
		if( inputFormat == kPGPInputFormat_PKCS12 )
		{

			/* Pick up optional passphrase */
			if( IsPGPError( err = pgpFindOptionArgs( optionList,
								kPGPOptionType_Passphrase, FALSE,
								"%p%l", &passphrase, &passphraseLength ) ) )
				goto error;

			err = PKCS12InputKey( context, bufPtr, bufLength,
								  (PGPByte *) passphrase, passphraseLength,
								  &outBuf, &outBufLength,
								  &certSet, &certSetLength );
			if( IsPGPError( err ) )
				goto error;
			
			/* Switch input to PKCS-8 data */
			if( mustFreeBuf )
				PGPFreeData( bufPtr );
			bufPtr = outBuf;
			bufLength = outBufLength;
			mustFreeBuf = TRUE;
		}

		/* Now have PKCS-8 data in bufPtr/bufLength */
		
		/* Process the returned cert set */
		if( IsntNull( certSet ) )
		{
			err = pgpDecodeX509CertSet( certSet, certSetLength,
										context, refdb, db );
			if( IsPGPError( err ) )
				goto error;
			refdb = *db;		/* Use just-imported cert as ref */
		}

		/* Decode PKCS-8 data */
		err = pgpDecodePCKS8( bufPtr, bufLength, context, refdb,
							  IsntNull(*db)?&db2:db );
		if( IsPGPError( err ) )
			goto error;

		/* Combine keysets if necessary */
		if( IsntNull( db2 ) )
		{
			err = PGPCopyKeys( pgpKeyDBPeekRootSet(db2), *db, NULL );
			if( IsPGPError( err ) )
				goto error;
			PGPFreeKeyDB( db2 );
			db2 = NULL;
		}

		/* Set passphrase on newly imported key */
		if( passphraseLength != 0 )
		{
			/* Find key we just imported */
			for( key = (*db)->firstKeyInDB; IsntNull(key); key=key->next )
			{
				if( !pgpKeyDBObjIsReal( key ) )
					continue;
				if (pgpKeyIsSec( key ) )
					break;
			}
			if( key )
			{
				err = pgpDoChangePassphrase_internal( *db, key, NULL, NULL, 0,
												passphrase, passphraseLength,
												FALSE, 0, FALSE );
				if( IsPGPError( err ) )
					goto error;
			}
		}

		/* Done */
		goto error;
	}

	if( inputFormat >= kPGPInputFormat_PEMEncodedX509Cert &&
		inputFormat <= kPGPInputFormat_NetscapeV1_PEMEncoded )
	{
		/* Need to remove PEM encoding */
		PGPByte *tmpBuf;
		PGPSize tmpBufLength;
		err = pgpRemovePEMEncoding( context, bufPtr, bufLength,
									&tmpBuf, &tmpBufLength );
		if( IsPGPError( err ) )
			goto error;
		/* Replace bufPtr, bufLength with tmp versions (which must be freed) */
		if( mustFreeBuf )
			PGPFreeData( bufPtr );
		mustFreeBuf = TRUE;
		bufPtr = tmpBuf;
		bufLength = tmpBufLength;
	}

	/* Process buffer to strip off any PKCS-7 layering */
	{
		PGPKeyDBObjRef dummy1, dummy2;
		PGPBoolean dummy3, dummy4, dummy5;
		PGPAttributeValue *dummy6;
		PGPUInt32 dummy7;

		err = X509InputCertificate( context, bufPtr, bufLength, refdb,
									inputFormat, NULL, &dummy1, &dummy2,
									&dummy3, &dummy4, &dummy5,
									&dummy6, &dummy7, &certSet, &certSetLength,
									&crlSet, &crlSetLength );
		if( IsPGPError( err ) )
			goto error;
		
		if( IsntNull( certSet ) )
		{
			err = pgpDecodeX509CertSet( certSet, certSetLength, context,
										refdb, db );
			if( IsPGPError( err ) )
				goto error;
		}

		if( IsntNull( crlSet ) )
		{
			err = pgpDecodeX509CRLSet( crlSet, crlSetLength, context, refdb,
									   IsntNull(*db)?&db2:db );
			if( IsPGPError( err ) )
				goto error;

			if( IsNull( *db ) )
			{
				err = kPGPError_X509NeededCertNotAvailable;
				goto error;
			}

			/* Combine keysets if necessary */
			if( IsntNull( db2 ) )
			{
				PGPKeySetRef crls;
				err = PGPCopyKeys( pgpKeyDBPeekRootSet(db2), *db, &crls );
				if( IsPGPError( err ) )
					goto error;
				PGPFreeKeyDB( db2 );
				db2 = NULL;
				/* Run a sig check to make CRL effective */
				PGPCheckKeyRingSigs( crls, NULL, FALSE, NULL, NULL );
			}
			else
			{
				/* Run a sig check to make CRL effective */
				PGPCheckKeyRingSigs( pgpKeyDBPeekRootSet(*db), NULL,
									 FALSE, NULL, NULL );
			}
		}
	}

error:
	if( mustFreeBuf )
		PGPFreeData( bufPtr );
	if( IsntNull( certSet ) )
		PGPFreeData( certSet );
	if( IsntNull( crlSet ) )
		PGPFreeData( crlSet );
	if( IsntNull( db2 ) )
		PGPFreeKeyDB( db2 );

	return err;
}



static const PGPOptionType impkeyOptionSet[] = {
	kPGPOptionType_InputFileRef,
	kPGPOptionType_LocalEncoding,
	kPGPOptionType_InputBuffer,
	kPGPOptionType_EventHandler,
	kPGPOptionType_SendNullEvents,
	kPGPOptionType_InputFormat,
	kPGPOptionType_Passphrase,
	kPGPOptionType_CachePassphrase,
	kPGPOptionType_X509Encoding,
	kPGPOptionType_KeyDBRef
};

/* Frees optionList, unlike most other internal functions */
	PGPError
pgpImportKeySetInternal (PGPContextRef context, PGPKeyDBRef *keydb,
	PGPOptionListRef optionList)
{
	PGPError		err = kPGPError_NoErr;
	PGPKeyDBRef		kdb;
	PGPInputFormat	inputFormat;
	PGPUInt32		fDo509;

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						impkeyOptionSet, elemsof( impkeyOptionSet ) ) ) )
		return err;

	pgpAssertAddrValid( keydb, PGPKeyDBRef );
	*keydb = NULL;
	
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_X509Encoding, FALSE,
						 "%d", &fDo509 ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_InputFormat, FALSE,
						 "%d", &inputFormat ) ) )
		goto error;

	if( fDo509 || inputFormat >= kPGPInputFormat_X509DataInPKCS7 )
	{
		err = sImportX509Certificate( context, keydb, inputFormat, optionList);
		PGPFreeOptionList( optionList );
		goto error;
	}

	if( IsPGPError( err = PGPNewKeyDB( context, &kdb ) ) )
		goto error;

	if( IsPGPError( err = PGPDecode( context, optionList,
									 PGPODiscardOutput(context, TRUE),
									 PGPOImportKeysTo(context, kdb),
									 PGPOLastOption(context) ) ) ) {
		PGPFreeKeyDB( kdb );
		goto error;
	}
	*keydb = kdb;
	
error:
	return err;
}


/*
 * Add our custom PGP X.509 extension, if useExtension is true.  Else add
 * the data to the description field.  Fill into *av.  buf is a working
 * buff at least 40 bytes long.  key tells which key to get the PGP data
 * from.
 */
	static void
sAddPGPX509Extension( PGPAttributeValue *av, char *buf, PGPKeyDBObjRef key,
					  PGPBoolean useExtension )
{
	static char			 s_pgpkeycr[] = "PGPKeyCreation=0x";
	static PGPByte		 s_pgpx509keycr [] = {
		0x30, 0x0e,		/* SEQUENCE */
			/* PGP Extension OID */
			/* (1 3 6 1 4 1 3401 8 1 1) */
			0x06, 0x0a, 0x2b, 0x06, 0x01, 0x04, 0x01, 0x9a, 0x49, 0x08,
					0x01, 0x01,
			0x04, 0x00	/* Octet string */
			/* Value as UTCTime value goes here */
	};

	if( useExtension )
	{
		/* Use a new extension */
		PGPUInt32 kcr = pgpKeyCreation (key);
		PGPByte *px = (PGPByte *)buf + sizeof(s_pgpx509keycr);
		PGPSize tlen;
		char tbuf[PGPX509TIMELEN+1];

		tlen = pgpTimeToX509Time( kcr, tbuf );
		pgpCopyMemory( s_pgpx509keycr, buf, sizeof(s_pgpx509keycr) );
		buf[1] += tlen + 2;
		buf[sizeof(s_pgpx509keycr)-1] += tlen + 2;
		/* Choose GeneralizedTime vs UTCTime tag */
		*px++ = (tlen == PGPX509TIMELEN) ? 24 : 23;
		*px++ =  tlen;
		pgpCopyMemory( tbuf, px, tlen );
		av->attribute = kPGPAVAttribute_CertificateExtension;
		av->size = sizeof(s_pgpx509keycr) + 2 + tlen;
		av->value.pointervalue = buf;
		av->unused = 0;
	}
	else
	{
		/* Use description field in subject name */
		pgpCopyMemory( s_pgpkeycr, buf, sizeof(s_pgpkeycr)-1 );
		sprintf ( buf+sizeof(s_pgpkeycr)-1, "%08x",
				  pgpKeyCreation (key) );
		av->attribute = kPGPAVAttribute_Description;
		av->size = sizeof(s_pgpkeycr)-1 + 8;
		av->value.pointervalue = buf;
		av->unused = 0;
	}
}



/*
 * Handle X509 based export formats.
 * Key is the top level key we are exporting; keyset controls which
 * sub objects if any we should look at
 */
static PGPError
sExportKeyX509 (
	PGPKeyDBObjRef	 	key,
	PGPKeySetRef		keyset,
	PGPExportFormat		exportFormat,
	PGPOptionListRef	optionList
	)
{
	PGPContextRef		context;
	PGPKeyDBRef			kdb;
	PGPError			err = kPGPError_NoErr;
	PGPByte				*buf = NULL;
	PGPSize				 bufLength = 0;
	PGPBoolean			 freeBuf = FALSE;
	PGPUInt32			 fExportSecrets;
	PGPMemoryMgrRef		 mgr;
	void				*vFormatData;
	PGPAttributeValue	*formatData;
	PGPAttributeValue	*formatDataCopy = NULL;
	PGPAttributeValue	*newAV;
	PGPSize				 formatDataLength;
	PGPOptionListRef	 passphrase = NULL;
	PGPByte				*dpoint = NULL;
	PGPSize				 dpointlen = 0;
	PGPInt32			 annotate = 0;
	char				 pgpkeycr[40];	/* big enough for either case */

	kdb = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( kdb );
	mgr = PGPPeekContextMemoryMgr( context );

	if (exportFormat == kPGPExportFormat_X509Cert) {
		/* Find the right X509 sig, somehow */
		PGPKeyDBObj *obj = NULL;
		PGPKeyDBObj *bestsig = NULL;
		PGPKeyIter *iter;

		if( IsPGPError( err = PGPNewKeyIterFromKeySet( keyset, &iter ) ) )
			goto error;
		while( IsntPGPError( pgpKeyIterNextObject( iter, &obj ) ) ) {
			if (pgpObjectType(obj) == RINGTYPE_SIG &&
				pgpSigIsX509 (obj)) {
				PGPKeyDBObj *signer = pgpSigMaker (obj);
				/* Use a self-sig if exists, else use first sig */
				if (signer == obj->up) {
					bestsig = obj;
				} else if (bestsig == NULL) {
					bestsig = obj;
				}
			}
		}
		PGPFreeKeyIter( iter );
		iter = NULL;
		if (bestsig != NULL) {
			buf = pgpSigX509Certificate( bestsig, &bufLength );
		}
		annotate = PGPANN_ARMORX509_CERTIFICATE;
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							 kPGPOptionType_ExportPrivateKeys, FALSE,
							 "%d", &fExportSecrets ) ) )
			goto error;
		if( fExportSecrets )
		{
			/* Do PKCS-12 conversion */
			PGPByte *password;
			PGPSize passwordLength;
			PGPByte *p8buf;
			PGPSize p8buflen;
			PGPByte *p12buf;
			PGPSize p12buflen;
			PGPByte *friendlyName;
			PGPSize friendlyLen;
			PGPSecKey *seckey;
			PGPKeyDBObjRef userID;

			if( IsPGPError( err = pgpFindOptionArgs( optionList,
								kPGPOptionType_Passphrase, FALSE,
								"%p%l", &password, &passwordLength ) ) )
				goto error;

			seckey = pgpSecSecKey (key, PGP_PKUSE_SIGN);
			if (!seckey)
			{
				err = pgpKeyDBError(PGPPeekKeyDBObjKeyDB(key));
				goto error;
			}
			err = pgpSecKeyUnlock( seckey, (char *) password, passwordLength, FALSE );
			if( err != 1 )
			{
				err = kPGPError_BadPassphrase;
				goto error;
			}
			err = pgpEncodePKCS8( seckey, &p8buf, &p8buflen );
			pgpSecKeyDestroy( seckey );
			if( IsPGPError( err ) )
				goto error;
			PGPGetPrimaryUserID( key, &userID );
			pgpGetUserIDAllocatedStringBuffer( userID, kPGPUserIDProperty_Name,
											   &friendlyName, &friendlyLen );
			err = pgpCreateCertSetFromChain( context, bestsig, &buf,
											 &bufLength );
			if( IsPGPError( err ) )
			{
				PGPFreeData( friendlyName );
				PGPFreeData( p8buf );
				goto error;
			}
			err = PKCS12ExportKey( context, p8buf, p8buflen, password,
								   passwordLength, buf, bufLength,
								   friendlyName, friendlyLen,
								   1, 1000, kPGPPBEAlgorithm_SHA3DES,
								   &p12buf, &p12buflen );
			PGPFreeData( buf );
			PGPFreeData( p8buf );
			PGPFreeData( friendlyName );
			if( IsPGPError( err ) )
				goto error;
			buf = p12buf;
			bufLength = p12buflen;
			freeBuf = TRUE;
			annotate = PGPANN_ARMORX509_PRIVATE;
		}
	} else if (exportFormat >= kPGPExportFormat_X509GetCRL) {
		time_t curtime = PGPGetStdTimeFromPGPTime( PGPGetTime() );

		/* If InputBuffer is specified, it is CRL distribution point */
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							 kPGPOptionType_InputBuffer, FALSE,
							 "%p%l", &dpoint, &dpointlen ) ) )
			goto error;

		err = X509CreateCRLRequest ( context, key, dpoint, dpointlen,
				exportFormat, curtime, &buf, &bufLength );
		if( IsPGPError( err ) )
			goto error;

		freeBuf = TRUE;
	} else if (exportFormat >= kPGPExportFormat_X509GetCertInitial) {
		/* Structure for now will be a fixed empty sequence */
		static PGPByte emptysequence[] = {0x30, 0x00};
		buf = emptysequence;
		bufLength = sizeof(emptysequence);
		freeBuf = FALSE;
	} else if (exportFormat >= kPGPExportFormat_X509CertReq) {
		PGPOption passop;

		if( IsPGPError( err = pgpSearchOptionSingle( optionList,
									kPGPOptionType_Passphrase, &passop ) ) )
			goto error;
		if( IsntOp( passop ) ) {
			if( IsPGPError( err = pgpSearchOptionSingle( optionList,
										kPGPOptionType_Passkey, &passop ) ) )
				goto error;
		}

		if( IsOp( passop ) ) {
			PGPOption passopcopy;
			pgpCopyOption( context, &passop, &passopcopy );
			passphrase = pgpNewOneOptionList( context, &passopcopy );
		}

		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							 kPGPOptionType_AttributeValue, FALSE,
							 "%p%l",
							 &vFormatData, &formatDataLength ) ) )
			goto error;
		formatData = vFormatData;
		formatDataLength /= sizeof(PGPAttributeValue);
		formatDataCopy = PGPNewData( mgr,
					(formatDataLength+1)*sizeof(PGPAttributeValue), 0 );
		if( IsNull( formatDataCopy ) ) {
			err = kPGPError_OutOfMemory;
			goto error;
		}
		pgpCopyMemory( formatData, formatDataCopy+1,
					   formatDataLength*sizeof(PGPAttributeValue) );
		newAV = formatDataCopy;

		/* Add "description" or extension field to hold keycreation data */
		sAddPGPX509Extension( newAV, pgpkeycr, key,
		  (PGPBoolean)(exportFormat==kPGPExportFormat_VerisignV1_CertReq ) );

		err = X509CreateCertificateRequest ( context, key, exportFormat,
				formatDataCopy, formatDataLength+1, passphrase,
				&buf, &bufLength );
		passphrase = NULL;
		PGPFreeData( formatDataCopy );
		formatDataCopy = NULL;
		if( IsPGPError( err ) )
			goto error;

		freeBuf = TRUE;
		annotate = PGPANN_ARMORX509_CERT_REQUEST;
	} else {
		pgpAssert (0);
	}

	err = pgpOutputBufferByOptionList( context, buf, bufLength, optionList, annotate );

error:

	if( IsntNull( passphrase ) )
		pgpFreeOptionList( passphrase );
	if( IsntNull( formatDataCopy ) )
		PGPFreeData( formatDataCopy );
	if( freeBuf && IsntNull( buf ) )
		PGPFreeData( buf );

	return err;
}


static const PGPOptionType cx509OptionSet[] = {
	kPGPOptionType_Passphrase,
	kPGPOptionType_Passkey,
	kPGPOptionType_CachePassphrase,
	kPGPOptionType_AttributeValue,
	kPGPOptionType_Expiration,
	kPGPOptionType_CreationDate

};


/*
 * Create X.509 certificate
 */
	PGPError
pgpCreateX509CertificateInternal(
	PGPKeyDBObjRef		signer,
	PGPKeyDBObjRef	 	targetkey,
	PGPKeyDBObjRef		*sig,
	PGPOptionListRef	optionList
	)
{
	PGPContextRef		context;
	PGPMemoryMgrRef		mgr;
	char *				passphrase;
	PGPSize				passphraseLength;
	PGPBoolean			hashedPhrase = FALSE;
	PGPTime				creationTime;
	PGPUInt32			expirationDays;
	PGPError			err;
	void				*vFormatData;
	PGPAttributeValue	*formatData;
	PGPAttributeValue	*formatDataCopy;
	PGPAttributeValue	*newAV;
	PGPSize				formatDataLength;
	char				 pgpkeycr[40];	/* big enough for custom ext. */

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						cx509OptionSet, elemsof( cx509OptionSet ) ) ) )
		return err;


	context = PGPPeekKeyDBObjContext( signer );
	mgr = PGPPeekContextMemoryMgr( context );

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_CreationDate, FALSE,
						 "%T", &creationTime ) ) )
		goto error;
		
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_Expiration, FALSE,
						 "%d", &expirationDays ) ) )
		goto error;
		
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_Passphrase, FALSE,
						"%p%l", &passphrase, &passphraseLength ) ) )
		goto error;
	if (IsNull( passphrase )) {
		hashedPhrase = TRUE;
		if( IsPGPError( err = pgpFindOptionArgs( optionList,
							kPGPOptionType_Passkey, FALSE,
							"%p%l", &passphrase, &passphraseLength ) ) )
			goto error;
	}

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_AttributeValue, TRUE,
						 "%p%l",
						 &vFormatData, &formatDataLength ) ) )
		goto error;
	formatData = vFormatData;
	formatDataCopy = PGPNewData( mgr,
				formatDataLength+sizeof(PGPAttributeValue), 0 );
	if( IsNull( formatDataCopy ) ) {
		err = kPGPError_OutOfMemory;
		goto error;
	}
	pgpCopyMemory( formatData, formatDataCopy+1, formatDataLength );
	newAV = formatDataCopy;
	formatDataLength += sizeof(PGPAttributeValue);

	/* Add extension field to hold keycreation data */
	sAddPGPX509Extension( newAV, pgpkeycr, targetkey, TRUE );

	err = pgpX509CreateCert_internal( context, signer, targetkey,
							creationTime, expirationDays, newAV,
							formatDataLength, (PGPByte *)passphrase,
							passphraseLength, hashedPhrase, sig );
	if( IsPGPError( err ) )
		goto error;


 error:

	return err;
}



/*
 * Export function.  Not clear how to split this between front end and
 * back end.  Either need to get data from back end to front end and do
 * it all on front end, or we need to be able to export objects one at
 * a time in back end, appending to a buffer we are accumulating.  No,
 * the first way is better.
 */
	PGPError
pgpExportToPGPFile( PGPKeySet *set, PGPFile *file,
	PGPBoolean exportmastersecrets, PGPBoolean exportsubsecrets,
	PGPBoolean includeattributes, PGPBoolean includepubkey )
{
	PGPKeyDBObj *		key;
	PGPKeyDBObj *		child;
	PGPKeyDBObj *		gchild;
	PGPByte const *		data;
	PGPSize				datalen;
	PGPKeyDB *			kdb = set->keyDB;
	PGPError			err = kPGPError_NoErr;

	for (key = kdb->firstKeyInDB; IsntNull(key); key = key->next) {
		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		if( !pgpKeySetIsMember( key, set ) )
			continue;
		if( includepubkey )  {	/*if the storage is precious, we may need to exclude it*/
			data = pgpFetchObject( key, &datalen );
			if (!exportmastersecrets && pgpKeyIsSec (key)) {
				data = pgpKeyDBObjToPubData( key, &datalen );
				if( pgpFileWrite( data, datalen, file ) != datalen ) {
					err = kPGPError_WriteFailed;
					goto error;
				}
				PGPFreeData( (PGPByte *)data );
			} else {
				if( pgpFileWrite( data, datalen, file ) != datalen ) {
					err = kPGPError_WriteFailed;
					goto error;
				}
			}
		}
		for (child = key->down; IsntNull(child); child = child->next) {
			if( !pgpKeyDBObjIsReal( child ) )
				continue;
			if( !pgpKeySetIsMember( child, set ) )
				continue;
			if (!includeattributes && pgpObjectType (child) == RINGTYPE_USERID
											&& pgpUserIDIsAttribute (child) )
				continue;
			if( !includeattributes && OBJISCRL(child) )
				continue;
			if( OBJISSIG(child) )
			{	PGPSigInfo *sinfo = pgpSigToSigInfo( child );
				if( !SIGISEXPORTABLE(sinfo) )
					continue;
				if( pgpSigIsSuperceded(child) )
					continue;
				if( exportmastersecrets )
					continue;		/* No sigs in private key files */
			}
			data = pgpFetchObject( child, &datalen );
			if (!exportsubsecrets && OBJISKEY(child) && pgpKeyIsSec (child)) {
				data = pgpKeyDBObjToPubData( child, &datalen );
				if( pgpFileWrite( data, datalen, file ) != datalen ) {
					err = kPGPError_WriteFailed;
					goto error;
				}
				PGPFreeData( (PGPByte *)data );
			} else {
				if( pgpFileWrite( data, datalen, file ) != datalen ) {
					err = kPGPError_WriteFailed;
					goto error;
				}
			}
			for (gchild = child->down; IsntNull(gchild);
				 								gchild = gchild->next) {
				if( !pgpKeyDBObjIsReal( gchild ) )
					continue;
				if( !pgpKeySetIsMember( gchild, set ) )
					continue;
				if( !includeattributes && OBJISCRL(gchild) )
					continue;
				if( OBJISSIG(gchild) )
				{	PGPSigInfo *sinfo = pgpSigToSigInfo( gchild );
					if( !SIGISEXPORTABLE(sinfo) )
						continue;
					if( pgpSigIsSuperceded(gchild) )
						continue;
					if( OBJISKEY(child)?exportsubsecrets:exportmastersecrets )
						continue;		/* No sigs in private key files */
				}
				data = pgpFetchObject( gchild, &datalen );
				if( pgpFileWrite( data, datalen, file ) != datalen ) {
					err = kPGPError_WriteFailed;
					goto error;
				}
			}
		}
	}
 error:
	return err;
}





#if PGP_MACINTOSH
#pragma global_optimizer on
#endif

static PGPError
sExportKeyContainer( PGPKeySetRef keyset, PGPBoolean mandatory, PGPOptionListRef optionList )  
{
	PGPError			err = kPGPError_NoErr;
    PGPContextRef       context;

    PGPByte             *contName = NULL;
    PGPSize             contSize=0;

	void				*password=NULL;
	PGPSize             passwordLength=0;
	void                *PIN=NULL;
	PGPSize             PINlength=0;
	PGPBoolean			hashedPassword, hashedPIN;

    PGPKeyDBObjRef key;

	pgpAssert( pgpKeySetIsValid( keyset ) );

    context = PGPPeekKeyDBContext( keyset->keyDB );

    if( IsNull( key = pgpFirstKeyInKeySet( keyset ) ) )
        goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_TokenKeyContainerSuggested, mandatory,
						 "%p%l", &contName, &contSize ) ) )
		goto error;

	pgpSearchPassphraseOptions( optionList, &password, &passwordLength,
								&hashedPassword, &PIN, &PINlength, &hashedPIN);
	if( password == NULL || hashedPIN )
	{
		err = kPGPError_BadParams;
		goto error;
	}
	
    if( IsntPGPError(err) && contSize )  
    {
        PGPKeyInfo *kinfo = pgpKeyToKeyInfo( key );

        err = pgpTokenPutKeyContainer_back( 
            context, kinfo->keyID, 
            (IsNull(PIN)?password:PIN),(IsNull(PIN)?passwordLength:PINlength), 
            contName, contSize
            );
    }

error:
    return err;
}

/* Exports certificate from the given keyset to the appropriate token */
static PGPError
sExportKeyX509ToToken ( PGPKeySetRef keyset, PGPOptionListRef optionList )
{
	PGPError			err = kPGPError_NoErr;

    PGPKeyDBObj *obj = NULL;
    PGPKeyDBObj *bestsig = NULL;
    PGPKeyIter *iter;

	PGPByte             *password=NULL;
	PGPSize             passwordLength=0;

    PGPKeyDBObjRef      key;
    PGPContextRef       context;

	PGPByte		        *buf = NULL;
	PGPSize		        bufLength = 0;

    PGPOAllocatedOutputBufferDesc	*contOutDesc = NULL;

	pgpAssert( pgpKeySetIsValid( keyset ) );

    key = pgpFirstKeyInKeySet( keyset );
    if( IsNull( key ) )  {
        err = kPGPError_BadParams;
        goto error;
    }

#if 0
	/* Key maybe in the temporary DB, so attributes will not be copied */
	pgpAssert( pgpKeyIsSec(key) );  
    if( ! pgpKeyIsOnToken( key ) )  
        return kPGPError_SecretKeyNotFound;
#endif

    context = PGPPeekKeyDBContext( keyset->keyDB );

    if( IsPGPError( err = pgpFindOptionArgs( optionList,
        kPGPOptionType_Passphrase, FALSE,
        "%p%l", &password, &passwordLength ) ) )
        goto error;

    /* We are working with token-based key here. */
#ifdef PGP_DEBUG
    {
        PGPBoolean          fOutputToken=FALSE;
        PGPUInt32           tokNumber=0;

        if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_OutputToken, FALSE,
						 "%b%d", &fOutputToken, &tokNumber ) ) )
        pgpAssert( fOutputToken && tokNumber == -1 ); 
            /* Output to the token corresponding to KeyID */
    }
#endif

    /* Find the right X509 sig, somehow */
    if( IsPGPError( err = PGPNewKeyIterFromKeySet( keyset, &iter ) ) )
        goto error;
    while( IsntPGPError( pgpKeyIterNextObject( iter, &obj ) ) ) {
        if (pgpObjectType(obj) == RINGTYPE_SIG &&
            pgpSigIsX509 (obj)) {
            PGPKeyDBObj *signer = pgpSigMaker (obj);
            /* Use a self-sig if exists, else use first sig */
            if (signer == obj->up) {
                bestsig = obj;
            } else if (bestsig == NULL) {
                bestsig = obj;
            }
        }
    }
    PGPFreeKeyIter( iter );
    iter = NULL;
    if (bestsig != NULL) {
        buf = pgpSigX509Certificate( bestsig, &bufLength );
    }

    /* Return output buffer with X509 cert, if was requested */
	{
		PGPOAllocatedOutputBufferDesc	*desc;

		err = pgpFindOptionArgs( optionList,
					kPGPOptionType_OutputAllocatedBuffer, FALSE, "%p", &desc );

		if( IsntNull( desc ) )
		{
	        void *outBuffer = 
                (void*)PGPNewData( 
                    PGPPeekContextMemoryMgr(context), 
                    bufLength, kPGPMemoryMgrFlags_None);
            if( outBuffer == NULL )   {
                err = kPGPError_OutOfMemory;
                goto error;
            }
            pgpCopyMemory(buf, outBuffer, bufLength);

			*desc->buffer = outBuffer;
			*desc->actualBufferSize = bufLength;
		}
	}

	/* Return existing key container, if was requested */
    if( IsPGPError( err = pgpFindOptionArgs( optionList,
            kPGPOptionType_TokenKeyContainer, FALSE,
            "%p", &contOutDesc ) ) )
        goto error;

    /* Load onto the token by calling backend */
    {
        PGPKeyInfo *kinfo = pgpKeyToKeyInfo( key );
        PGPKeyDBObjRef userID;
		PGPByte *pUserID;
		PGPSize pUserID_len;
        const PGPKeyDBObjProperty prop = kPGPUserIDProperty_CommonName; 
            /* kPGPUserIDProperty_Name is too long and may not 
               be appropriate for other viewers. */

		PGPGetPrimaryUserID( key, &userID );
		pgpGetUserIDAllocatedStringBuffer( userID, prop,
											   &pUserID, &pUserID_len );

        err = pgpTokenImportX509_back( context, kinfo->keyID, 
                   pUserID,      pUserID_len, 
                   buf,          bufLength, 
                   password,     passwordLength );

        PGPFreeData( pUserID );


		if( (IsntPGPError(err) || err == kPGPError_SmartCardX509Exists) &&
			contOutDesc )  
        {
            err = pgpTokenGetKeyContainer_back( 
                context, kinfo->keyID, 
				password, passwordLength,
                (PGPByte **)contOutDesc->buffer, contOutDesc->actualBufferSize);
        }
    }


error:
    return err;
}


/*
 * Frees optionList, unlike most other internal functions.  rset
 * defines the exact set to be exported; key is the first key in the
 * set.
 */
static PGPError
sExportKeySetInternal (PGPKeySet *keyset, PGPOptionListRef optionList)
{
	PGPContextRef		context;
	PGPKeyDBRef			kdb;
	PGPUInt32			fExportSecrets;
	PGPUInt32			fExportSubSecrets;
	PGPBoolean			fExportAttributes;
	PGPBoolean			fExportFormat;

    /* For export of key or X509 cert to the token */
    PGPBoolean			fOutputToken;

	PGPByte				*buf;
	PGPSize				bufSize;
	PGPSize				bufSizeRead;
	PGPFile				*pfile;
	PGPOptionListRef	optList;
	PGPExportFormat		exportFormat = kPGPExportFormat_Complete;
	PGPUInt32			wExportFormat;
	PGPBoolean			armorop;
	PGPUInt32			nkeys;
	PGPError			err = kPGPError_NoErr;

	pgpAssert( pgpKeySetIsValid( keyset ) );
	kdb = keyset->keyDB;
	context = PGPPeekKeyDBContext( kdb );

	buf = NULL;
	pfile = NULL;
	optList = NULL;

	/* Read optional options */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_ExportPrivateKeys, FALSE,
						 "%d", &fExportSecrets ) ) )
		goto error;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_ExportPrivateSubkeys, FALSE,
						 "%d", &fExportSubSecrets ) ) )
		goto error;

	/* Clear exportSecrets flags if doing a single non-secret key */
	if( (PGPCountKeys(keyset, &nkeys), nkeys == 1) &&
		!pgpKeyIsSec( pgpFirstKeyInKeySet( keyset ) ) )
	{
		fExportSecrets = FALSE;
		fExportSubSecrets = FALSE;
	}

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_ExportFormat, FALSE,
						 "%b%d", &fExportFormat, &wExportFormat ) ) )
		goto error;

	if (fExportFormat)
		exportFormat = (PGPExportFormat) wExportFormat;

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_OutputToken, FALSE,
						 "%b", &fOutputToken ) ) )
		goto error;

    /* Put private key on the token */
    if( fOutputToken && exportFormat == kPGPExportFormat_Complete )  {
        err = pgpCopyKeyToToken( keyset, optionList );
        if( IsntPGPError(err) )
            err = sExportKeyContainer( keyset, FALSE, optionList );
        goto error;
    }

	/* Handle X509 export formats */
	if (exportFormat == kPGPExportFormat_X509Cert ||
		exportFormat >= kPGPExportFormat_X509CertReq) {
		PGPKeyDBObjRef key;
		if( IsNull( key = pgpFirstKeyInKeySet( keyset ) ) )
			goto error;
        if( fOutputToken )  {
            /* Export X509 certificate to the token */
            err = sExportKeyX509ToToken( keyset, optionList );

            /* Export key container, if specified */
            if( IsntPGPError(err) )
                err = sExportKeyContainer( keyset, FALSE, optionList );
        }
        else
		    err = sExportKeyX509 (key, keyset, exportFormat, optionList);
		goto error;
	}

    /* Handle optional key container */
    if( fOutputToken && IsntPGPError(err) && 
        exportFormat == kPGPExportFormat_TokenKeyContainer )  
    {
        /* Export key container, must be specified */
        err = sExportKeyContainer( keyset, TRUE, optionList );
        goto error;
    }

	fExportAttributes = (exportFormat > kPGPExportFormat_Basic);

	/* Output public or private portion */

	/* Create memory buffer to write to */
	pfile = pgpFileMemOpen( context, NULL, 0 );
	if( IsNull( pfile ) ) {
		err = kPGPError_OutOfMemory;
		goto error;
	}

	/* Output data to memory buffer */
	if( IsPGPError( err = pgpExportToPGPFile( keyset, pfile,
								(PGPBoolean)fExportSecrets,
								(PGPBoolean)(fExportSubSecrets|fExportSecrets),
								fExportAttributes, TRUE /*includepubkey*/ ) ) )
		goto error;

	/* Read data we just wrote (hopefully smaller than 2 GB -wjb) */
	bufSize = (PGPSize)pgpFileTell (pfile );
	buf = (PGPByte *)pgpContextMemAlloc( context, bufSize, 0 );
	if( IsNull( buf ) ) {
		err = kPGPError_OutOfMemory;
		goto error;
	}
	(void)pgpFileSeek( pfile, 0, SEEK_SET );
	bufSizeRead = pgpFileRead( buf, bufSize, pfile );
	pgpAssert( bufSizeRead == bufSize );
	pgpFileClose( pfile );
	pfile = NULL;

	/* Do ascii armoring */
	/* If user specified an ascii armor option, use his, else use TRUE */
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						kPGPOptionType_ArmorOutput, FALSE,
						"%b", &armorop ) ) )
		goto error;

	/* This next call frees optionList */
	if( IsPGPError( err = PGPBuildOptionList( context, &optList, optionList, 
									 (armorop ?
									  PGPONullOption(context) :
									  PGPOArmorOutput(context, TRUE)),
									 PGPORawPGPInput(context, TRUE),
									 PGPOCompression(context, FALSE),
									 PGPOLastOption(context) ) ) )
		goto error;
	optionList = NULL;

	if( IsPGPError( err = PGPEncode( context, optList,
									 PGPOInputBuffer(context, buf, bufSize),
									 PGPOLastOption(context) ) ) )
		goto error;

	pgpContextMemFree( context, buf );
	buf = NULL;

	/*
	 * If exporting private keys, append public keys to the buffer.
	 * We can't export them both together, as we end up with an output
	 * which has private keys followed by names and sigs.  This can not
	 * be safely imported, because of the "version bug".  The private key
	 * may have the incorrect version.  So we instead output the public
	 * part with names and sigs, after the private part with private keys
	 * and names.
	 */
	if( fExportSecrets || fExportSubSecrets ) {

		/* Create memory buffer to write to */
		pfile = pgpFileMemOpen( context, NULL, 0 );
		if( IsNull( pfile ) ) {
			err = kPGPError_OutOfMemory;
			goto error;
		}

		/* Output data to memory buffer */
		if( IsPGPError( err = pgpExportToPGPFile( keyset, pfile, FALSE, FALSE,
										fExportAttributes, TRUE /*includepubkey*/ ) ) )
			goto error;

		/* Read data we just wrote (hopefully smaller than 2 GB -wjb)*/
		bufSize = (PGPSize)pgpFileTell (pfile );
		buf = (PGPByte *)pgpContextMemAlloc( context, bufSize, 0 );
		if( IsNull( buf ) ) {
			err = kPGPError_OutOfMemory;
			goto error;
		}
		(void)pgpFileSeek( pfile, 0, SEEK_SET );
		bufSizeRead = pgpFileRead( buf, bufSize, pfile );
		pgpAssert( bufSizeRead == bufSize );
		pgpFileClose( pfile );
		pfile = NULL;

		/* Do ascii armoring, append to existing output */
		PGPAppendOptionList( optList,
							 PGPOAppendOutput(context, TRUE),
							 PGPOLastOption(context) );
		if( IsPGPError( err = PGPEncode( context, optList,
									PGPOInputBuffer(context, buf, bufSize),
									PGPOLastOption(context) ) ) )
			goto error;

		pgpContextMemFree( context, buf );
		buf = NULL;
	}

	err = kPGPError_NoErr;

	/* Fall through */
error:
	if( IsntNull( optList ) )
		PGPFreeOptionList( optList );
	if( IsntNull( optionList ) )
		PGPFreeOptionList( optionList );
	if( IsntNull( pfile ) )
		pgpFileClose( pfile );
	if( IsntNull( buf ) )
		pgpContextMemFree( context, buf );
	return err;
}

#if PGP_MACINTOSH
#pragma global_optimizer reset
#endif


static const PGPOptionType expkeyOptionSet[] = {
	kPGPOptionType_ExportPrivateKeys,
	kPGPOptionType_ExportPrivateSubkeys,
	kPGPOptionType_ExportFormat,
	kPGPOptionType_OutputFileRef,
	kPGPOptionType_OutputBuffer,
	kPGPOptionType_OutputAllocatedBuffer,
	kPGPOptionType_DiscardOutput,
	kPGPOptionType_ArmorOutput,
	kPGPOptionType_CommentString,
	kPGPOptionType_VersionString,
	kPGPOptionType_EventHandler,
	kPGPOptionType_SendNullEvents,
	kPGPOptionType_OutputLineEndType,
	kPGPOptionType_OutputFormat,
	/* Used for cert requests */
	kPGPOptionType_Passphrase,
	kPGPOptionType_Passkey,
	kPGPOptionType_CachePassphrase,
	kPGPOptionType_AttributeValue,
	kPGPOptionType_InputBuffer
};



/* Frees optionList, unlike most other internal functions. */
PGPError
pgpExportKeySetInternal (PGPKeySet *keys, PGPOptionListRef optionList)
{
	PGPError			err = kPGPError_NoErr;

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						expkeyOptionSet, elemsof( expkeyOptionSet ) ) ) )
		return err;

	err = sExportKeySetInternal (keys, optionList );

	return err;

}


static const PGPOptionType expOptionSet[] = {
	kPGPOptionType_ExportPrivateKeys,
	kPGPOptionType_ExportPrivateSubkeys,
	kPGPOptionType_ExportFormat,
	kPGPOptionType_OutputFileRef,
	kPGPOptionType_OutputBuffer,
	kPGPOptionType_OutputAllocatedBuffer,
	kPGPOptionType_DiscardOutput,
	kPGPOptionType_ArmorOutput,
	kPGPOptionType_CommentString,
	kPGPOptionType_VersionString,
	kPGPOptionType_EventHandler,
	kPGPOptionType_SendNullEvents,
	kPGPOptionType_OutputLineEndType,
	kPGPOptionType_OutputFormat,
	kPGPOptionType_ExportKeySet,
	kPGPOptionType_ExportKeyDBObj,
	/* Used for cert requests */
	kPGPOptionType_Passphrase,
	kPGPOptionType_Passkey,
	kPGPOptionType_CachePassphrase,
	kPGPOptionType_AttributeValue,
	kPGPOptionType_InputBuffer,
    /* Used for Token operation */
    kPGPOptionType_OutputToken,
    kPGPOptionType_TokenKeyContainer,
    kPGPOptionType_TokenKeyContainerSuggested
};



/*
 * Frees optionList, unlike most other internal functions.  This is
 * like exportkeyset, but it allows a single userid or sig to be
 * specified.  We only export that object, plus its parent object(s)
 * and all its children.  This is especially convenient when exporting
 * a particular X.509 certificate.
 */
PGPError
pgpExportInternal (PGPContextRef context, PGPOptionListRef optionList)
{
	PGPKeyDBObj		   *obj;
	PGPKeySet		   *keys = NULL;
	PGPBoolean			fExportKeyDBObjRef;
	PGPBoolean			fExportKeySetRef;
	PGPBoolean			fDeleteKeys = FALSE;
	void			   *wExportKeyDBObjRef;
	void			   *wExportKeySetRef;
	PGPInt32			successes = 0;
	PGPError			err = kPGPError_NoErr;

	(void) context;

	if (IsPGPError( err = pgpCheckOptionsInSet( optionList,
						expOptionSet, elemsof( expOptionSet ) ) ) )
		return err;


	/* See if we have a target key/name/sig to export */

	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_ExportKeySet, FALSE,
						 "%b%d", &fExportKeySetRef, &wExportKeySetRef ) ) )
		goto error;
	if (fExportKeySetRef)
		++successes;
	if( IsPGPError( err = pgpFindOptionArgs( optionList,
						 kPGPOptionType_ExportKeyDBObj, FALSE,
						 "%b%d", &fExportKeyDBObjRef, &wExportKeyDBObjRef ) ) )
		goto error;
	if (fExportKeyDBObjRef)
		++successes;
	
	if (successes > 1) {
		pgpDebugMsg( "too many key object selection options for PGPExport" );
		err = kPGPError_BadParams;
		goto error;
	} else if (successes == 0) {
		pgpDebugMsg( "no key object selection options for PGPExport" );
		err = kPGPError_BadParams;
		goto error;
	}

	/* Create keyset we will export */

	/* Handle the different cases */
	if (fExportKeySetRef) {
		keys = (PGPKeySet *) wExportKeySetRef;
	} else {
		obj = (PGPKeyDBObj *) wExportKeyDBObjRef;
		if( IsPGPError( err = PGPNewOneKeySet( obj, &keys ) ) )
			goto error;
		fDeleteKeys = TRUE;
	}

	if( IsPGPError( err = sExportKeySetInternal (keys, optionList ) ) )
		goto error;

	/* Fall through */
error:
	if( fDeleteKeys && IsntNull( keys ) )
		PGPFreeKeySet( keys );

	return err;
}


/*
 * Local Variables:
 * tab-width: 4
 * End:
 * vi: ts=4 sw=4
 * vim: si
 */
